/****************************************************************************
 *   Copyright (C) 2009 by Claas Anders "CaScAdE" Rathje                    *
 *   admiralcascade@gmail.com                                               *
 *   Project-URL: http://www.mylifesucks.de/oss/c-osd/                      *
 *                                                                          *
 *   This program is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by   *
 *   the Free Software Foundation; either version 2 of the License.         *
 *                                                                          *
 *   This program is distributed in the hope that it will be useful,        *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *   GNU General Public License for more details.                           *
 *                                                                          *
 *   You should have received a copy of the GNU General Public License      *
 *   along with this program; if not, write to the                          *
 *   Free Software Foundation, Inc.,                                        *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 ****************************************************************************/

#include <avr/io.h>
#include <util/delay.h>
#include <avr/pgmspace.h> 
#include "max7456_software_spi.h"

/* ##########################################################################
 * MAX7456 SPI & Display stuff
 * ##########################################################################*/

/**
 * Send a byte through SPI
 */
void spi_send(uint8_t byte) {
    for (int8_t i = 7; i >= 0; i--) {
        if (((byte >> i) & 1)) {
            MAX_SDIN_HIGH
        } else {
            MAX_SDIN_LOW
        }
        MAX_SCLK_HIGH
        MAX_SCLK_LOW
    }
}

/**
 *  Send <byte> to <address> of MAX7456
 */
void spi_send_byte(uint8_t address, uint8_t byte) {
    // start sending
    MAX_CS_LOW

    spi_send(address);
    spi_send(byte);

    // end sending
    MAX_CS_HIGH
}

/**
 *  write a <character> to <address> of MAX7456 display memory
 */
void write_char(uint16_t address, char character) {
    spi_send_byte(0x05, (address & 0xFF00) >> 8); // DMAH
    spi_send_byte(0x06, (address & 0x00FF)); // DMAL
    spi_send_byte(0x07, character); // DMDI
}

/**
 *  write a character <attribute> to <address> of MAX7456 display memory
 */
void write_char_att(uint16_t address, char attribute) {
    // the only important part is that the DMAH[1] is set
    // so we add 2 which binary is the 2nd lowest byte
    spi_send_byte(0x05, ((address & 0xFF00) >> 8) | 2); // DMAH
    spi_send_byte(0x06, (address & 0x00FF)); // DMAL
    spi_send_byte(0x07, attribute); // DMDI
}

/**
 *  write a <character> at <x>/<y> to MAX7456 display memory
 */
void write_char_xy(uint8_t x, uint8_t y, char character) {
    uint16_t address = y * 30 + x;
    write_char(address, character);
}

/**
 *  write a  character <attribute> at <x>/<y> to MAX7456 display memory
 */
void write_char_att_xy(uint8_t x, uint8_t y, char attribute) {
    uint16_t address = y * 30 + x;
    write_char_att(address, attribute);
}

/**
 *  clear display by writing blank characters all over it
 */
void clear(void) {
    uint16_t memory_address = 0;
    for (unsigned int a = 0; a < 480; a++) {
        write_char(memory_address++, 0);
    }
}

/**
 *  write an ascii <character> to <address> of MAX7456 display memory
 */
void write_ascii_char(uint16_t address, char c) {
    if (c == 32) c = 0; // remap space
    else if (c > 48 && c <= 57) c -= 48; // remap numbers
    else if (c == '0') c = 10; // remap zero
    else if (c >= 65 && c <= 90) c -= 54; // remap big letters
    else if (c >= 97 && c <= 122) c -= 60; // remap small letters
    else if (c == '(') c = 63; // remap
    else if (c == ')') c = 64; // remap
    else if (c == '.') c = 65; // remap
    else if (c == '?') c = 66; // remap
    else if (c == ';') c = 67; // remap
    else if (c == ':') c = 68; // remap
    else if (c == ',') c = 69; // remap
    else if (c == '\'') c = 70; // remap
    else if (c == '/') c = 71; // remap
    else if (c == '"') c = 72; // remap
    else if (c == '-') c = 73; // remap minus
    else if (c == '<') c = 74; // remap
    else if (c == '>') c = 75; // remap
    else if (c == '@') c = 76; // remap
    write_char(address, c);
}

/**
 *  write an ascii <string> at <x>/<y> to MAX7456 display memory
 */
void write_ascii_string(uint8_t x, uint8_t y, char *string) {
    while (*string) {
        write_ascii_char(((x++)+(y * 30)), *string);
        string++;
    }
}

/**
 *  write an ascii <string> from progmen at <x>/<y> to MAX7456 display memory
 */
void write_ascii_string_pgm(uint8_t x, uint8_t y, char *string) {
 	while (pgm_read_byte(string) != 0x00) 
		write_ascii_char(((x++)+(y * 30)), pgm_read_byte(string++));
}

/**
 * Write only some digits of a unsigned <number> at <x>/<y> to MAX7456 display memory
 * <num> represents the largest multiple of 10 that will still be displayable as
 * the first digit, so num = 10 will be 0-99 and so on
 * <pad> = 1 will cause blank spaced to be filled up with zeros e.g. 007 instead of   7
 */
void write_ndigit_number_u(uint8_t x, uint8_t y, uint16_t number, int16_t num, uint8_t pad) {
		// if number is largar than 99[..]9 we must decrease it
		while (number >= (num * 10)) {
			number -= num * 10;
		}

		uint8_t started = 0;

		while (num > 0) {
			uint8_t b = number / num;
			if (b > 0 || started || num == 1) {
			    write_ascii_char((x++)+(y * 30), '0' + b);
			    started = 1;
			} else {
				if (pad) write_ascii_char((x++)+(y * 30), '0');
				else write_ascii_char((x++)+(y * 30), 0);
			}
			number -= b * num;

			num /= 10;
		}
}

/**
 * Write only some digits of a signed <number> at <x>/<y> to MAX7456 display memory
 * <num> represents the largest multiple of 10 that will still be displayable as
 * the first digit, so num = 10 will be 0-99 and so on
 * <pad> = 1 will cause blank spaced to be filled up with zeros e.g. 007 instead of   7
 */
void write_ndigit_number_s(uint8_t x, uint8_t y, int16_t number, int16_t num, uint8_t pad) {
    if (((uint16_t) number) > 32767) {
        number = number - 65536;
		num *= -1;

		// if number is smaller than -99[..]9 we must increase it
		while (number <= (num * 10)) {
			number -= num * 10;
		}

		uint8_t started = 0;

		while (num < 0) {
			uint8_t b = number / num;
			if (pad) write_ascii_char((x)+(y * 30), '0');
			if (b > 0 || started || num == 1) {
				if (!started) write_char((x - 1)+(y * 30), 0x49);
			    write_ascii_char((x++)+(y * 30), '0' + b);
			    started = 1;
			} else {
				write_ascii_char((x++)+(y * 30), 0);
			}
			number -= b * num;

			num /= 10;
		}
	} else {
        write_char((x)+(y * 30), 0);
        write_ndigit_number_u(x, y, number, num, pad);
    }
}

/**
 * Write only some digits of a unsigned <number> at <x>/<y> to MAX7456 display memory
 * as /10th of the value
 * <num> represents the largest multiple of 10 that will still be displayable as
 * the first digit, so num = 10 will be 0-99 and so on
 * <pad> = 1 will cause blank spaced to be filled up with zeros e.g. 007 instead of   7
 */
void write_ndigit_number_u_10th(uint8_t x, uint8_t y, uint16_t number, int16_t num, uint8_t pad) {
		// if number is largar than 99[..]9 we must decrease it
		while (number >= (num * 10)) {
			number -= num * 10;
		}
		

		uint8_t started = 0;
		while (num > 0) {
			uint8_t b = number / num;
			if (b > 0 || started || num == 1) {
				if ((num / 10) == 0) {
					if (!started) write_ascii_char((x - 1)+(y * 30), '0');
					write_char((x++)+(y * 30), 65); // decimal point
				}
				write_ascii_char((x++)+(y * 30), '0' + b);
			    started = 1;
			} else {
				if (pad) write_ascii_char((x++)+(y * 30), '0');
				else write_ascii_char((x++)+(y * 30), ' ');
			}
			number -= b * num;

			num /= 10;
		}
}

/**
 * Write only some digits of a signed <number> at <x>/<y> to MAX7456 display memory
 * as /10th of the value
 * <num> represents the largest multiple of 10 that will still be displayable as
 * the first digit, so num = 10 will be 0-99 and so on
 * <pad> = 1 will cause blank spaced to be filled up with zeros e.g. 007 instead of   7
 */
void write_ndigit_number_s_10th(uint8_t x, uint8_t y, int16_t number, int16_t num, uint8_t pad) {
    if (((uint16_t) number) > 32767) {
        number = number - 65536;
		num *= -1;

		// if number is smaller than -99[..]9 we must increase it
		while (number <= (num * 10)) {
			number -= num * 10;
		}

		uint8_t started = 0;

		while (num < 0) {
			uint8_t b = number / num;
			if (pad) write_ascii_char((x)+(y * 30), '0');
			if (b > 0 || started || num == 1) {
				if ((num / 10) == 0) {
					if (!started) {
						write_ascii_char((x - 2)+(y * 30), '-');
						write_ascii_char((x - 1)+(y * 30), '0');
					}
					write_char((x++)+(y * 30), 65); // decimal point
				} else if (!started) {
					write_char((x - 1)+(y * 30), 0x49); // minus
				}
			    write_ascii_char((x++)+(y * 30), '0' + b);
			    started = 1;
			} else {
				write_ascii_char((x++)+(y * 30), 0);
			}
			number -= b * num;

			num /= 10;
		}
	} else {
        write_char((x)+(y * 30), 0);
        write_ndigit_number_u_10th(x, y, number, num, pad);
    }
}

/**
 *  write <seconds> as human readable time at <x>/<y> to MAX7456 display mem
 */
void write_time(uint8_t x, uint8_t y, uint16_t seconds) {
    uint16_t min = seconds / 60;
    seconds -= min * 60;
    write_ndigit_number_u(x, y, min, 100, 0);
    write_char_xy(x + 3, y, 68);
    write_ndigit_number_u(x + 4, y, seconds, 10, 1);
}

/**
 * for testing write all chars to screen
 */
void write_all_chars() {
    uint16_t x = 3, y = 2, t = 0;
    while (t < 256) {
        write_char_xy(x++, y, t++);
        if (x > 25) {
            y++;
            x = 3;
        }
    }
}

/**
 * let the MAX7456 learn a new character at <number>
 * with <data>.
 */
void learn_char(uint8_t number, unsigned char* data) {
    // select character to write (CMAH)
    spi_send_byte(0x09, number);

    for (uint8_t i = 0; i < 54; i++) {
        // select 4pixel byte of char (CMAL)
        spi_send_byte(0x0A, i);

        // write 4pixel byte of char (CMDI)
        spi_send_byte(0x0B, data[i]);
    }

    // write to the NVM array from the shadow RAM (CMM)
    spi_send_byte(0x08, 0b10100000);

    // according to maxim writing to nvram takes about 12ms, lets wait longer
    _delay_ms(120);
}
