/****************************************************************************
 *   Copyright (C) 2009 by Claas Anders "CaScAdE" Rathje                    *
 *   admiralcascade@gmail.com                                               *
 *   Project-URL: http://www.mylifesucks.de/oss/c-osd/                      *
 *                                                                          *
 *   This program is free software; you can redistribute it and/or modify   *
 *   it under the terms of the GNU General Public License as published by   *
 *   the Free Software Foundation; either version 2 of the License.         *
 *                                                                          *
 *   This program is distributed in the hope that it will be useful,        *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *   GNU General Public License for more details.                           *
 *                                                                          *
 *   You should have received a copy of the GNU General Public License      *
 *   along with this program; if not, write to the                          *
 *   Free Software Foundation, Inc.,                                        *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.              *
 *                                                                          *
 *                                                                          *
 *   Credits to:                                                            *
 *   Holger Buss & Ingo Busker from mikrokopter.de for the MK project       *
 *   Gregor "killagreg" Stobrawa for making the MK code readable            *
 *   Klaus "akku" Buettner for the hardware                                 *
 *   Manuel "KeyOz" Schrape for explaining the MK protocol to me            *
 ****************************************************************************/

#include <avr/io.h>
#include <avr/interrupt.h>
#include <util/delay.h>
#include <avr/pgmspace.h> 
#include "main.h"
#include "max7456_software_spi.h"
#include "usart1.h"

/* TODO:
 * - verifiy correctness of values
 * - clean up code :)
 */

#if !(ALLCHARSDEBUG|(WRITECHARS != -1))
// data structs not needed for character flashin
#include "mk-data-structs.h"

/* ##########################################################################
 * global definitions and global vars
 * ##########################################################################*/

volatile uint16_t setsReceived = 0;

volatile NaviData_t naviData;
volatile DebugOut_t debugData;

// cache old vars for blinking attribute, checkup is faster than full
// attribute write each time
volatile uint8_t last_UBat = 255;
volatile uint8_t last_RC_Quality = 255;

// 16bit should be enough, normal LiPos don't last that long
volatile uint16_t uptime = 0;
volatile uint16_t timer = 0;

// remember last time data was received
volatile uint8_t seconds_since_last_data = 0;

// store stats description in progmem to save space
char stats_item_0[] PROGMEM = "max Altitude:";
char stats_item_1[] PROGMEM = "max Speed   :";
char stats_item_2[] PROGMEM = "max Distance:";
char stats_item_3[] PROGMEM = "min voltage :";
char stats_item_4[] PROGMEM = "max time    :";
char* stats_item_pointers[] PROGMEM = {stats_item_0, stats_item_1, stats_item_2, stats_item_3, stats_item_4};

// store more fixed strings in progmen
char ON[]  PROGMEM = "ON ";
char OFF[] PROGMEM = "OFF";

#endif // ends !(ALLCHARSDEBUG|(WRITECHARS != -1))

// general PAL|NTSC distingiusch stuff
uint8_t top_line = 1;
uint8_t bottom_line = 14;

// Flags
uint8_t COSD_FLAGS = 0;

/* ##########################################################################
 * debounce buttons
 * ##########################################################################*/
int s1_pressed() {
	if (S1_PRESSED) {
		_delay_ms(25);
		if (S1_PRESSED) return 1;
	}
	return 0;
}

int s2_pressed() {
	if (S2_PRESSED) {
		_delay_ms(25);
		if (S2_PRESSED) return 1;
	}
	return 0;
}


#if !(ALLCHARSDEBUG|(WRITECHARS != -1))

/**
 * serial support
 */
//#include "usart1.c"


/* ##########################################################################
 * timer stuff
 * ##########################################################################*/

/**
 * timer kicks in every 1000uS ^= 1ms
 */
ISR(TIMER0_OVF_vect) {
    OCR0 = 6; // preload
    if (!timer--) {
        uptime++;
        timer = 999;
		seconds_since_last_data++;
    }
}

/* ##########################################################################
 * compass stuff
 * ##########################################################################*/

/**
 * convert the <heading> gotton from NC into an index
 */
uint8_t heading_conv(uint16_t heading) {
    if (heading > 23 && heading < 68) {
        //direction = "NE";
        return 0;
    } else if (heading > 67 && heading < 113) {
        //direction = "E ";
        return 1;
    } else if (heading > 112 && heading < 158) {
        //direction = "SE";
        return 2;
    } else if (heading > 157 && heading < 203) {
        //direction = "S ";
        return 3;
    } else if (heading > 202 && heading < 248) {
        //direction = "SW";
        return 4;
    } else if (heading > 247 && heading < 293) {
        //direction = "W ";
        return 5;
    } else if (heading > 292 && heading < 338) {
        //direction = "NW";
        return 6;
    }
    //direction = "N ";
    return 7;
}

/**
 * draw a compass rose at <x>/<y> for <heading>
 */
void draw_compass(uint8_t x, uint8_t y, uint16_t heading) {
    //char* rose = "---N---O---S---W---N---O---S---W---N---O---S---W";
    char rose[48] = {216, 215, 216, 211, 216, 215, 216, 213, 216, 215, 216, 212,
                    216, 215, 216, 214, 216, 215, 216, 211, 216, 215, 216, 213,
                    216, 215, 216, 212, 216, 215, 216, 214, 216, 215, 216, 211,
                    216, 215, 216, 213, 216, 215, 216, 212, 216, 215, 216, 214};
	// the center is char 19 (north), we add the current heading in 8th
	// which would be 22.5 degrees, but float would bloat up the code
	// and *10 / 225 would take ages... so we take the uncorrect way
    uint8_t front = 19 + (heading / 22);
    for (uint8_t i = 0; i < 9; i++) {
		write_char_xy(x++, y, rose[front - 4 + i]);
    }
}

/* ##########################################################################
 * artificial horizon
 * ##########################################################################*/
// remember last time displayed values
int8_t old_af_x = -1, old_af_y = -1;

/**
 * draw roll und nick indicators (could be enhanced to full artificial horizon)
 * from line <firstline> to <listlines> for given <nick> and <roll> values
 */
void draw_artificial_horizon(uint8_t firstline, uint8_t lastline, int16_t nick, int16_t roll) {
	char noodle[5] = {225, 225, 226, 227, 227};
	uint8_t center_x = 15;
	uint8_t center_y = lastline - firstline;
	center_y = 7;
	write_char_xy(center_x,center_y,228);
	uint8_t cpos, nicky, rollx;
	
	// which line
	int8_t ypos =  nick / 20;
	// which character from the array?
	if (nick < 0) {
		cpos = -1*((nick - (ypos * 20))/4);
		ypos--;
	} else cpos = 4-((nick - (ypos * 20))/4);
	if (cpos > 4) cpos = 4;

	nicky = center_y - ypos;
	if (nicky > lastline) nicky = lastline;
	else if (nicky < firstline) nicky = firstline;

	// ensure roll-borders
	rollx = (roll / 8)+15;
	if (rollx < 2) rollx = 2;
	else if (rollx > 28) rollx = 28;


	// clear roll
	if (old_af_x != rollx && old_af_x >= 0) {
		write_char_xy(old_af_x,13,0);
	}

	// clear nick
	if (old_af_y != nicky && old_af_y >= 0) {
		write_char_xy(center_x-1,old_af_y,0);
		write_char_xy(center_x+1,old_af_y,0);
	}


	// draw nick
	write_char_xy(center_x-1,nicky,noodle[cpos]);
	write_char_xy(center_x+1,nicky,noodle[cpos]);

	// draw roll
	write_char_xy(rollx,lastline,229);

	// update old vars
	old_af_x = rollx;
	old_af_y = nicky;

	// debug numbers
	//write_3digit_number_u(20,6,cpos);
	//write_number_s(20,7,ypos);	
	//write_number_s(0,7,nick);		
	//write_number_s(18,11,roll);	
}

/* ##########################################################################
 * A simple config menu for the flags
 * ##########################################################################*/

/**
 * helper function for menu updating
 */
void config_menu_drawings(uint8_t chosen) {
    // clear prevoius _cursor_
    write_ascii_string(3, (chosen + 6) % 7, " ");
    // draw current _cursor_
    write_ascii_string(3, chosen + 6, ">");
    if (COSD_FLAGS & COSD_FLAG_HUD) {
        write_ascii_string_pgm(23, 6, ON);
    } else {
		write_ascii_string_pgm(23, 6, OFF);
    }
    if (COSD_FLAGS & COSD_FLAG_ARTHORIZON) {
        write_ascii_string_pgm(23, 7, ON);
    } else {
        write_ascii_string_pgm(23, 7, OFF);
    }
	if (COSD_FLAGS & COSD_FLAG_STATS) {
        write_ascii_string_pgm(23, 8, ON);
    } else {
        write_ascii_string_pgm(23, 8, OFF);
    }
    if (COSD_FLAGS & COSD_FLAG_WARNINGS) {
        write_ascii_string_pgm(23, 9, ON);
    } else {
        write_ascii_string_pgm(23, 9, OFF);
    }
}

/**
 * some sort of clicking response in the menu
 */
void config_menu_doclick(uint8_t chosen, char** menu) {
	write_ascii_string(4, chosen + 6, "DONE              ");
	_delay_ms(1000);
	write_ascii_string(4, chosen + 6, menu[chosen]);
}

/**
 * a simple config menu tryout
 */
void config_menu(void) {
	// disable interrupts (makes the menu more smoothely)
	cli();

	// clear screen
	clear();
	
	char* menu[8] = {"Full HUD",
					 "Art.Horizon in HUD",
					 "Statistics",
					 "Warnings",	// TODO: do it!
					 "Reset uptime",
					 "Request OSD-data",
					 "Disable Debug-data",
					 "EXIT"};

	uint8_t inmenu = 1;
	uint8_t chosen = 0;
	write_ascii_string(6,  2, "C-OSD Config Menu");

	// wait a bit before doing stuff so user has chance to release button
	_delay_ms(250);

	write_ascii_string(4,  6, menu[0]);
	write_ascii_string(4,  7, menu[1]);
	write_ascii_string(4,  8, menu[2]);
	write_ascii_string(4,  9, menu[3]);
	write_ascii_string(4, 10, menu[4]);
	write_ascii_string(4, 11, menu[5]);
	write_ascii_string(4, 12, menu[6]);
	write_ascii_string(4, 13, menu[7]);

	config_menu_drawings(chosen);

	while (inmenu) {
			if (s2_pressed()) {
				write_ascii_string(3,  chosen+6, " ");
				chosen = (chosen + 1) % 8;
				write_ascii_string(3,  chosen+6, ">");
				_delay_ms(500);
			} else if (s1_pressed()) {
				switch (chosen) {
					case 0:		// full HUD
						COSD_FLAGS ^= COSD_FLAG_HUD;
						config_menu_drawings(chosen);
						break;
					case 1:		// art horizon
						COSD_FLAGS ^= COSD_FLAG_ARTHORIZON;
						config_menu_drawings(chosen);
						break;
					case 2:		// statistics
						COSD_FLAGS ^= COSD_FLAG_STATS;
						config_menu_drawings(chosen);
						break;
					case 3:		// warnings
						COSD_FLAGS ^= COSD_FLAG_WARNINGS;
						config_menu_drawings(chosen);
						break;
					case 4:		// reset uptime
						uptime = 0;
						config_menu_doclick(chosen, menu);
						break;
					case 5:		// re-request OSD data
						// request OSD Data from NC every 100ms
						usart1_request_mk_data(1, 'o', 100);
						config_menu_doclick(chosen, menu);
						break;
					case 6:		// disable debug data
						// disable sending of debug data
						// may result in smoother ddata display
						usart1_request_mk_data(0, 'd', 0);
						config_menu_doclick(chosen, menu);
						break;
					case 7:		// exit
						inmenu = 0;
						break;
				}
				_delay_ms(250);
			}
	}

	// clear screen up again
	clear();

	// update flags to paint display again if needed
	COSD_FLAGS &= ~COSD_ICONS_WRITTEN;

	// enable interrupts again
	sei();
}

#endif // ends !(ALLCHARSDEBUG|(WRITECHARS != -1))
/* ##########################################################################
 * MAIN
 * ##########################################################################*/
int main(void) {
    // set up FLAGS, compiler should flatten this one
    COSD_FLAGS = (NTSC << 0);
    COSD_FLAGS |= (HUD << 1);
    COSD_FLAGS |= (ARTHORIZON << 2);
	COSD_FLAGS |= (STATS << 3);
    COSD_FLAGS |= (WARNINGS << 4);

    // set up Atmega162 Ports
    DDRA |= (1 << PA1); // PA1 output (/CS)
    MAX_CS_HIGH
    DDRA |= (1 << PA2); // PA2 output (SDIN)
    MAX_SDIN_LOW
    DDRA |= (1 << PA3); // PA3 output (SCLK)
    MAX_SCLK_LOW
    DDRA |= (1 << PA5); // PA5 output (RESET)
    MAX_RESET_HIGH

    DDRC |= (1 << PC0); // PC0 output (LED1 gn)
    LED1_OFF
    DDRC |= (1 << PC1); // PC1 output (LED2 rt)
    LED2_OFF
    DDRC |= (1 << PC2); // PC2 output (LED3 gn)
    LED3_OFF
    DDRC |= (1 << PC3); // PC3 output (LED4 rt)
    LED4_OFF

    DDRC &= ~(1 << PC4); // PC4 input  (MODE)
    PORTC |= (1 << PC4); // pullup
    DDRC &= ~(1 << PC5); // PC5 input  (SET)
    PORTC |= (1 << PC5); // pullup

    // set up top and bottom lines
    if (COSD_FLAGS & COSD_FLAG_NTSC) {
        bottom_line = 12;
    } else {
        bottom_line = 14;
    }

    // reset the MAX7456 to be sure any undefined states do no harm
    MAX_RESET_LOW
    MAX_RESET_HIGH

    // give the FC/NC and the maxim time to come up
    LED4_ON
    _delay_ms(2000);

    LED4_OFF


    //Pushing NEW chars to the MAX7456
#if (WRITECHARS != -1)
    // DISABLE display (VM0)
    spi_send_byte(0x00, 0b00000000);
#include "characters.c"

#endif 

    // Setup Video Mode
    if (COSD_FLAGS & COSD_FLAG_NTSC) {
        // NTSC + enable display immediately (VM0)
        spi_send_byte(0x00, 0b00001000);
    } else {
        // PAL + enable display immediately (VM0)
        spi_send_byte(0x00, 0b01001000);
    }

    // clear all display-mem (DMM)
    spi_send_byte(0x04, 0b00000100);

    // clearing takes 12uS according to maxim so lets wait longer
    _delay_us(120);

    // 8bit mode
    spi_send_byte(0x04, 0b01000000);

    // write blank chars to whole screen
    clear();

#if !(ALLCHARSDEBUG|(WRITECHARS != -1))
    // init usart
    usart1_init();

    // set up timer
    TCCR0 |= (1 << CS00) | (1 << CS01); // timer0 prescaler 64
    OCR0 = 6; // preload
    TIMSK |= (1 << TOIE0); // enable overflow timer0

    // enable interrupts
    sei();
#endif

    //write_ascii_string(2,  7, "         CaScAdE          ");
    //write_ascii_string(2,  8, "is TESTING his open source");
    //write_ascii_string(2,  9, "    EPi OSD Firmware");

    // custom char preview
    /*write_char_xy( 2, 7, 200);
    write_char_xy( 3, 7, 201);
    write_char_xy( 4, 7, 202);
    write_char_xy( 5, 7, 203);
    write_char_xy( 6, 7, 204);
    write_char_xy( 7, 7, 205);
    write_char_xy( 8, 7, 206);
    write_char_xy( 9, 7, 207);
    write_char_xy(10, 7, 208);
    write_char_xy(11, 7, 209);
    write_char_xy(12, 7, 210);
    write_char_xy(13, 7, 211);
    write_char_xy(14, 7, 212);
    write_char_xy(15, 7, 213);
    write_char_xy(16, 7, 214);
    write_char_xy(17, 7, 215);*/

    // we are ready
    LED3_ON



#if ALLCHARSDEBUG | (WRITECHARS != -1)
	clear();
    write_all_chars();
#else
	// clear serial screen
	//usart1_puts("\x1B[2J\x1B[H");
	//usart1_puts("hello world!\r\n");


	// request data ever 100ms from FC;
	//usart1_request_mk_data(0, 'd', 100);

	// request OSD Data from NC every 100ms
	usart1_request_mk_data(1, 'o', 100);

    // and disable debug...
	//usart1_request_mk_data(0, 'd', 0);

    // disable TXD-pin
    usart1_DisableTXD();

    // stats for after flight
    int16_t max_Altimeter = 0;
    uint16_t max_GroundSpeed = 0;
    int16_t max_Distance = 0;
    uint8_t min_UBat = 255;
    uint16_t max_FlyingTime = 0;

    // flags from last round to check for changes
    uint8_t old_MKFlags = 0;

    char* directions[8] = {"NE", "E ", "SE", "S ", "SW", "W ", "NW", "N "};
    char arrowdir[8] = {218, 217, 224, 223, 222, 221, 220, 219};

	/*write_ndigit_number_s_10th(5, 10, 1, 100, 0);
	write_ndigit_number_s_10th(5, 11, -1, 100, 0);
	write_ndigit_number_s_10th(5, 12, -11, 100, 0);*/

    while (1) {
        // write icons at init or after menu/mode-switch
        if (!(COSD_FLAGS & COSD_ICONS_WRITTEN) && (COSD_FLAGS & COSD_FLAG_HUD)) {
            write_char_xy(5, top_line, 203); // km/h
            write_char_xy(10, top_line, 202); // RC-transmitter
            write_char_xy(16, top_line, 208); // degree symbol
            write_char_xy(27, top_line, 204); // small meters m
            write_ascii_string(6, bottom_line, "V"); // voltage
            write_char_xy(14, bottom_line, 209); // on clock
            write_char_xy(22, bottom_line, 210); // fly clock
            write_char_xy(26, bottom_line, 200); // sat1
            write_char_xy(27, bottom_line, 201); // sat2
            COSD_FLAGS |= COSD_ICONS_WRITTEN;
        }
        if (rxd_buffer_locked) {
            if (COSD_FLAGS & COSD_FLAG_HUD) {
                if (rxd_buffer[2] == 'D') { // FC Data
                    /*Decode64();
                    debugData = *((DebugOut_t*) pRxData);
                    write_number_s(12, 2, RxDataLen);
                    write_number_s(20, 2, setsReceived++);
                    write_number_s(12, 3, debugData.Analog[0]); // AngleNick
                    write_number_s(12, 4, debugData.Analog[1]); // AngleRoll
					write_number_s(12, 5, debugData.Analog[5]); // Height
                    write_number_s(12, 6, debugData.Analog[9]); // Voltage
                    write_number_s(12, 7, debugData.Analog[10]);// RC Signal
					write_number_s(12, 8, debugData.Analog[11]);// Gyro compass*/
                } else if (rxd_buffer[2] == 'O') { // NC OSD Data
                    Decode64();
                    naviData = *((NaviData_t*) pRxData);

                    // first line
                    write_ndigit_number_u(2, top_line, (uint16_t) (((uint32_t) naviData.GroundSpeed * 36) / 1000), 100, 0);

                    write_ndigit_number_u(7, top_line, naviData.RC_Quality, 100, 0);
                    if (naviData.RC_Quality <= RCLVL_WRN && last_RC_Quality > RCLVL_WRN) {
                        for (uint8_t x = 0; x < 4; x++)
                            write_char_att_xy(7 + x, top_line, BLINK);
                    } else if (naviData.RC_Quality > RCLVL_WRN && last_RC_Quality <= RCLVL_WRN) {
                        for (uint8_t x = 0; x < 4; x++)
                            write_char_att_xy(7 + x, top_line, 0);
                    }
                    last_RC_Quality = naviData.RC_Quality;

                    write_ndigit_number_u(13, top_line, naviData.CompassHeading, 100, 0);

                    write_ascii_string(17, top_line, directions[heading_conv(naviData.CompassHeading)]);

                    if (naviData.Variometer == 0) {
                        write_char_xy(20, top_line, 206); // plain line
                    } else if (naviData.Variometer > 0 && naviData.Variometer <= 10) {
                        write_char_xy(20, top_line, 234); // small arrow up
                    } else if (naviData.Variometer > 10) {
                        write_char_xy(20, top_line, 235); // big arrow up
                    } else if (naviData.Variometer < 0 && naviData.Variometer >= -10) {
                        write_char_xy(20, top_line, 232); // small arrow down
                    } else {
                        write_char_xy(20, top_line, 233); //big arrow down
                    }

                    //note:lephisto:according to several sources it's /30
                    if (naviData.Altimeter > 300 || naviData.Altimeter < -300) {
                        // above 10m only write full meters
                        write_ndigit_number_s(23, top_line, naviData.Altimeter / 30, 1000, 0);
                    } else {
                        // up to 10m write meters.dm
                        //write_number_u_10th(21, top_line, naviData.Altimeter / 3);
						write_ndigit_number_s_10th(23, top_line, naviData.Altimeter / 3, 100, 0);
                    }

                    // seccond line
                    draw_compass(11, top_line + 1, naviData.CompassHeading);

                    // TODO: verify correctness
                    uint16_t heading_home = (naviData.HomePositionDeviation.Bearing + 360 - naviData.CompassHeading) % 360;
                    write_char_xy(27, top_line + 1, arrowdir[heading_conv(heading_home)]);

                    write_ndigit_number_u(24, top_line + 1, naviData.HomePositionDeviation.Distance / 10, 100, 0);

                    // center
                    if (naviData.MKFlags & FLAG_MOTOR_RUN) { // should be engines running
                        if (!(old_MKFlags & FLAG_MOTOR_RUN)) { // motors just started, clear middle
                            clear();
                            // update flags to paint display again if needed
                            COSD_FLAGS &= ~COSD_ICONS_WRITTEN;
                        }
                        if (COSD_FLAGS & COSD_FLAG_ARTHORIZON) {
                            draw_artificial_horizon(top_line + 2, bottom_line - 1, naviData.AngleNick, naviData.AngleRoll);
                        }
                    } else {
						// stats
						if (COSD_FLAGS & COSD_FLAG_STATS) {
							write_ascii_string_pgm(2, 5, stats_item_pointers[0]); // max Altitude
	                        write_ndigit_number_s(18, 5, max_Altimeter / 30, 1000, 0);
	                        write_char_xy(22, 5, 204); // small meters m
							write_ascii_string_pgm(2, 6, stats_item_pointers[1]); // max Speed
	                        write_ndigit_number_u(19, 6, (uint16_t) (((uint32_t) max_GroundSpeed * 36) / 1000), 100, 0);
	                        write_char_xy(22, 6, 203); // km/h
							write_ascii_string_pgm(2, 7, stats_item_pointers[2]); // max Distance
	                        write_ndigit_number_u(19, 7, max_Distance / 100, 100, 0);
	                        write_char_xy(22, 7, 204); // small meters m
							write_ascii_string_pgm(2, 8, stats_item_pointers[3]); // min voltage
							write_ndigit_number_u_10th(18, 8, min_UBat, 100, 0);
	                        write_ascii_string(22, 8, "V"); // voltage
							write_ascii_string_pgm(2, 9, stats_item_pointers[4]); // max time
	                        write_time(16, 9, max_FlyingTime);
	                        write_char_xy(22, 9, 210); // fly clock
						} else if (COSD_FLAGS & COSD_FLAG_ARTHORIZON) { // if no stats there is space horizon
                            draw_artificial_horizon(top_line + 2, bottom_line - 1, naviData.AngleNick, naviData.AngleRoll);
                        }
                    }

                    // bottom line
					write_ndigit_number_u_10th(2, bottom_line, naviData.UBat, 100, 0);
                    if (naviData.UBat <= UBAT_WRN && last_UBat > UBAT_WRN) {
                        for (uint8_t x = 0; x < 7; x++)
                            write_char_att_xy(x, bottom_line, BLINK);
                    } else {
                        for (uint8_t x = 0; x < 7; x++)
                            write_char_att_xy(x, bottom_line, 0);
                    }

                    write_time(8, bottom_line, uptime);
                    write_time(16, bottom_line, naviData.FlyingTime);

                    write_ndigit_number_u(24, bottom_line, naviData.SatsInUse, 10, 0);

                    if (naviData.NCFlags & NC_FLAG_CH) {
                        write_char_xy(27, bottom_line, 231); // gps ch
                    } else if (naviData.NCFlags & NC_FLAG_PH) {
                        write_char_xy(27, bottom_line, 230); // gps ph
                    } else { // (naviData.NCFlags & NC_FLAG_FREE)
                        write_char_xy(27, bottom_line, 201); // sat2 (free)
                    }

                    //write_number_s(8, 5, RxDataLen);
                    //write_number_s(16, 5, setsReceived++);

                    // remember statistics
                    if (naviData.Altimeter > max_Altimeter) max_Altimeter = naviData.Altimeter;
                    if (naviData.GroundSpeed > max_GroundSpeed) max_GroundSpeed = naviData.GroundSpeed;
                    if (naviData.HomePositionDeviation.Distance > max_Distance) {
                        max_Distance = naviData.HomePositionDeviation.Distance;
                    }
                    if (naviData.UBat < min_UBat) min_UBat = naviData.UBat;
                    if (naviData.FlyingTime > max_FlyingTime) max_FlyingTime = naviData.FlyingTime;

                    old_MKFlags = naviData.MKFlags;
                }
            }
            seconds_since_last_data = 0;
            rxd_buffer_locked = 0;
        }
        // handle keypress
        if (s1_pressed()) {
            config_menu();
        }
        if (seconds_since_last_data > 2) {
			// request OSD Data from NC every 100ms
			usart1_request_mk_data(1, 'o', 100);
			seconds_since_last_data = 0;
        }
    }
#endif
    return 0;
}
